<?php
declare(strict_types=1);

// race.php — komplette, korrigierte Version
// Erwartet eine PDO-Instanz in db.php -> $pdo
require_once __DIR__ . '/db.php';

// PDO hart auf Exceptions stellen (falls nicht schon geschehen)
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
$pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);

/**
 * Renn-Details inkl. Pferden
 */
function getRace(PDO $pdo, int $raceId): ?array {
    $stmt = $pdo->prepare("
        SELECT r.id, r.name, r.starts_at, r.settled_at, r.winner_horse_id,
               h.id AS horse_id, h.name AS horse_name
        FROM races r
        LEFT JOIN race_horses rh ON rh.race_id = r.id
        LEFT JOIN horses h ON h.id = rh.horse_id
        WHERE r.id = ?
        ORDER BY h.name ASC
    ");
    $stmt->execute([$raceId]);
    $rows = $stmt->fetchAll();
    if (!$rows) return null;

    $race = [
        'id'             => (int)$rows[0]['id'],
        'name'           => (string)$rows[0]['name'],
        'starts_at'      => $rows[0]['starts_at'],
        'settled_at'     => $rows[0]['settled_at'],
        'winner_horse_id'=> $rows[0]['winner_horse_id'] !== null ? (int)$rows[0]['winner_horse_id'] : null,
        'horses'         => [],
    ];
    foreach ($rows as $r) {
        if ($r['horse_id'] !== null) {
            $race['horses'][] = [
                'horse_id'   => (int)$r['horse_id'],
                'horse_name' => (string)$r['horse_name'],
            ];
        }
    }
    return $race;
}

/**
 * Alle Wetten zu einem Rennen
 */
function getBetsForRace(PDO $pdo, int $raceId): array {
    $stmt = $pdo->prepare("
        SELECT b.id, b.user_id, u.username,
               b.horse_id, h.name AS horse_name,
               b.amount, b.status, b.payout, b.settled_at
        FROM bets b
        JOIN users  u ON u.id = b.user_id
        JOIN horses h ON h.id = b.horse_id
        WHERE b.race_id = ?
        ORDER BY b.id ASC
    ");
    $stmt->execute([$raceId]);
    return $stmt->fetchAll();
}

/**
 * Rennen abrechnen (Pari-Mutuel, kein House-Cut)
 * Setzt:
 *  - Gewinner: status=won, payout>0, settled_at=NOW()
 *  - Verlierer: status=lost, payout=0, settled_at=NOW()
 *  - races.winner_horse_id, races.settled_at
 */
function settleRaceBets(PDO $pdo, int $raceId, int $winnerHorseId): array {
    // Summen holen (WICHTIG: fetchColumn statt execute()-Rückgabe!)
    $stmtTotal = $pdo->prepare("
        SELECT IFNULL(SUM(amount),0) FROM bets
        WHERE race_id=? AND status='placed'
    ");
    $stmtTotal->execute([$raceId]);
    $pool = (float)$stmtTotal->fetchColumn();

    $stmtWinner = $pdo->prepare("
        SELECT IFNULL(SUM(amount),0) FROM bets
        WHERE race_id=? AND horse_id=? AND status='placed'
    ");
    $stmtWinner->execute([$raceId, $winnerHorseId]);
    $onWinner = (float)$stmtWinner->fetchColumn();

    $info = [
        'pool'          => $pool,
        'on_winner'     => $onWinner,
        'payout_factor' => 0.0,
        'updated'       => [],
    ];

    try {
        $pdo->beginTransaction();

        if ($pool <= 0.0 || $onWinner <= 0.0) {
            // Keine Einsätze oder niemand auf Sieger -> alles verliert
            $pdo->prepare("
                UPDATE bets
                SET status='lost', payout=0, settled_at=NOW()
                WHERE race_id=? AND status='placed'
            ")->execute([$raceId]);
        } else {
            $payoutFactor = $pool / $onWinner;
            $info['payout_factor'] = $payoutFactor;

            // Gewinner holen
            $winnersStmt = $pdo->prepare("
                SELECT id, user_id, amount
                FROM bets
                WHERE race_id=? AND horse_id=? AND status='placed'
            ");
            $winnersStmt->execute([$raceId, $winnerHorseId]);

            $updBet  = $pdo->prepare("
                UPDATE bets
                SET status='won', payout=?, settled_at=NOW()
                WHERE id=?
            ");
            $updUser = $pdo->prepare("
                UPDATE users
                SET balance = balance + ?
                WHERE id=?
            ");

            foreach ($winnersStmt as $b) {
                $betId  = (int)$b['id'];
                $userId = (int)$b['user_id'];
                $amount = (float)$b['amount'];
                $payout = round($amount * $payoutFactor, 2);

                $updBet->execute([$payout, $betId]);
                $updUser->execute([$payout, $userId]);

                $info['updated'][] = [
                    'bet_id'  => $betId,
                    'user_id' => $userId,
                    'payout'  => $payout,
                ];
            }

            // Verlierer markieren
            $pdo->prepare("
                UPDATE bets
                SET status='lost', payout=0, settled_at=NOW()
                WHERE race_id=? AND horse_id<>? AND status='placed'
            ")->execute([$raceId, $winnerHorseId]);
        }

        // Rennen markieren
        $pdo->prepare("
            UPDATE races
            SET settled_at=NOW(), winner_horse_id=?
            WHERE id=?
        ")->execute([$winnerHorseId, $raceId]);

        $pdo->commit();
        return $info;
    } catch (Throwable $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        throw $e;
    }
}

/* ===========================
 * CLI-Modus:
 *   php race.php settle <raceId> <winnerHorseId>
 *   php race.php show   <raceId>
 * =========================== */
if (PHP_SAPI === 'cli') {
    $action = $argv[1] ?? '';
    if ($action === 'settle') {
        $raceId = isset($argv[2]) ? (int)$argv[2] : 0;
        $winId  = isset($argv[3]) ? (int)$argv[3] : 0;
        if ($raceId > 0 && $winId > 0) {
            $info = settleRaceBets($pdo, $raceId, $winId);
            echo "Race $raceId settled. Winner horse $winId\n";
            echo "Pool: {$info['pool']} | OnWinner: {$info['on_winner']} | Factor: {$info['payout_factor']}\n";
            foreach ($info['updated'] as $u) {
                echo "- Bet {$u['bet_id']} -> User {$u['user_id']} +{$u['payout']}\n";
            }
            exit(0);
        }
        fwrite(STDERR, "Usage: php race.php settle <raceId> <winnerHorseId>\n");
        exit(1);
    } elseif ($action === 'show') {
        $raceId = isset($argv[2]) ? (int)$argv[2] : 0;
        if ($raceId > 0) {
            $race = getRace($pdo, $raceId);
            $bets = getBetsForRace($pdo, $raceId);
            if (!$race) {
                fwrite(STDERR, "Race not found\n");
                exit(1);
            }
            echo "Race #{$race['id']} {$race['name']}\n";
            echo "Starts: {$race['starts_at']} | Settled: " . ($race['settled_at'] ?? 'no') .
                 " | Winner: " . ($race['winner_horse_id'] ?? 'n/a') . "\n";
            echo "Horses:\n";
            foreach ($race['horses'] as $h) echo " - {$h['horse_id']}: {$h['horse_name']}\n";
            echo "Bets:\n";
            foreach ($bets as $b) {
                echo " #{$b['id']} U{$b['user_id']}({$b['username']}) "
                   . "H{$b['horse_id']}({$b['horse_name']}) "
                   . "Amt {$b['amount']} Status {$b['status']} Payout {$b['payout']}\n";
            }
            exit(0);
        }
        fwrite(STDERR, "Usage: php race.php show <raceId>\n");
        exit(1);
    } else {
        fwrite(STDERR, "Usage:\n  php race.php settle <raceId> <winnerHorseId>\n  php race.php show <raceId>\n");
        exit(1);
    }
}

/* ===========================
 * HTTP-Modus (leichtgewichtig)
 *  GET ?action=settle&race_id=12&winner_horse_id=3
 *  GET ?action=show&race_id=12
 * =========================== */
$action = $_GET['action'] ?? '';
if ($action === 'settle') {
    $raceId = (int)($_GET['race_id'] ?? 0);
    $winId  = (int)($_GET['winner_horse_id'] ?? 0);
    header('Content-Type: application/json; charset=utf-8');
    if ($raceId > 0 && $winId > 0) {
        try {
            $info = settleRaceBets($pdo, $raceId, $winId);
            echo json_encode([
                'ok'            => true,
                'race_id'       => $raceId,
                'winner_horse'  => $winId,
                'pool'          => $info['pool'],
                'on_winner'     => $info['on_winner'],
                'payout_factor' => $info['payout_factor'],
                'updated'       => $info['updated'],
            ], JSON_PRETTY_PRINT);
        } catch (Throwable $e) {
            http_response_code(500);
            echo json_encode(['ok'=>false,'error'=>$e->getMessage()]);
        }
    } else {
        http_response_code(400);
        echo json_encode(['ok'=>false,'error'=>'missing race_id or winner_horse_id']);
    }
    exit;
}

if ($action === 'show') {
    $raceId = (int)($_GET['race_id'] ?? 0);
    header('Content-Type: application/json; charset=utf-8');
    if ($raceId > 0) {
        $race = getRace($pdo, $raceId);
        $bets = getBetsForRace($pdo, $raceId);
        echo json_encode(['ok'=> (bool)$race, 'race'=>$race, 'bets'=>$bets], JSON_PRETTY_PRINT);
    } else {
        http_response_code(400);
        echo json_encode(['ok'=>false,'error'=>'missing race_id']);
    }
    exit;
}

// Fallback: einfache Textausgabe
header('Content-Type: text/plain; charset=utf-8');
echo "race3.php online\n";
echo "Use:\n";
echo "  ?action=show&race_id=12\n";
echo "  ?action=settle&race_id=12&winner_horse_id=3\n";
