<?php
// race.php
// Zuständig für Renn-Infos und Auszahlungen

require_once __DIR__ . '/db.php'; // $pdo = new PDO(...);

// -----------------------------------------------------------------------------
// Hilfsfunktionen
// -----------------------------------------------------------------------------

function getRace(PDO $pdo, int $raceId): ?array {
    $stmt = $pdo->prepare("
        SELECT r.id, r.name, r.starts_at, r.settled_at,
               h.id AS horse_id, h.name AS horse_name
        FROM races r
        LEFT JOIN race_horses rh ON rh.race_id = r.id
        LEFT JOIN horses h ON h.id = rh.horse_id
        WHERE r.id = ?
        ORDER BY h.name ASC
    ");
    $stmt->execute([$raceId]);

    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (!$rows) {
        return null;
    }

    // race meta aus erster zeile
    $race = [
        'id'         => (int)$rows[0]['id'],
        'name'       => $rows[0]['name'],
        'starts_at'  => $rows[0]['starts_at'],
        'settled_at' => $rows[0]['settled_at'],
        'horses'     => [],
    ];

    foreach ($rows as $r) {
        if ($r['horse_id'] !== null) {
            $race['horses'][] = [
                'horse_id'   => (int)$r['horse_id'],
                'horse_name' => $r['horse_name'],
            ];
        }
    }

    return $race;
}

function getBetsForRace(PDO $pdo, int $raceId): array {
    $stmt = $pdo->prepare("
        SELECT b.id, b.user_id, u.username,
               b.horse_id, h.name AS horse_name,
               b.amount, b.status, b.payout
        FROM bets b
        JOIN users u     ON u.id = b.user_id
        JOIN horses h    ON h.id = b.horse_id
        WHERE b.race_id = ?
        ORDER BY b.id ASC
    ");
    $stmt->execute([$raceId]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// -----------------------------------------------------------------------------
// Auszahl-Logik (gefixt)
// -----------------------------------------------------------------------------

function settleRaceBets(PDO $pdo, int $raceId, int $winnerHorseId): array {
    // Pool gesamt (alle Einsätze im Rennen, egal auf welches Pferd, Status 'placed')
    $stmtTotal = $pdo->prepare("
        SELECT IFNULL(SUM(amount),0)
        FROM bets
        WHERE race_id = ? AND status='placed'
    ");
    $stmtTotal->execute([$raceId]);
    $pool = (float)$stmtTotal->fetchColumn();

    // Summe der Einsätze auf den Sieger
    $stmtWinner = $pdo->prepare("
        SELECT IFNULL(SUM(amount),0)
        FROM bets
        WHERE race_id = ? AND horse_id = ? AND status='placed'
    ");
    $stmtWinner->execute([$raceId, $winnerHorseId]);
    $onWinner = (float)$stmtWinner->fetchColumn();

    $pdo->beginTransaction();

    $result = [
        'pool'         => $pool,
        'on_winner'    => $onWinner,
        'payoutFactor' => 0.0,
        'updated'      => [],
    ];

    if ($pool <= 0.0) {
        // Es gab keine Wetten -> alles verlieren mit 0 Auszahlung
        $pdo->prepare("
            UPDATE bets
            SET status='lost', payout=0, settled_at=NOW()
            WHERE race_id=? AND status='placed'
        ")->execute([$raceId]);

    } elseif ($onWinner <= 0.0) {
        // Niemand hat auf das Siegerpferd gesetzt
        $pdo->prepare("
            UPDATE bets
            SET status='lost', payout=0, settled_at=NOW()
            WHERE race_id=? AND status='placed'
        ")->execute([$raceId]);

    } else {
        // klassischer Totalisator ohne House-Cut
        $payoutFactor = $pool / $onWinner;
        $result['payoutFactor'] = $payoutFactor;

        // Gewinner-Bets holen
        $winnersStmt = $pdo->prepare("
            SELECT id, user_id, amount
            FROM bets
            WHERE race_id=? AND horse_id=? AND status='placed'
        ");
        $winnersStmt->execute([$raceId, $winnerHorseId]);

        $updateBetStmt = $pdo->prepare("
            UPDATE bets
            SET status='won', payout=?, settled_at=NOW()
            WHERE id=?
        ");

        $updateUserStmt = $pdo->prepare("
            UPDATE users
            SET balance = balance + ?
            WHERE id=?
        ");

        foreach ($winnersStmt as $betRow) {
            $betId   = (int)$betRow['id'];
            $userId  = (int)$betRow['user_id'];
            $amount  = (float)$betRow['amount'];

            $payout  = round($amount * $payoutFactor, 2);

            // Bet kennzeichnen
            $updateBetStmt->execute([$payout, $betId]);

            // Guthaben gutschreiben
            $updateUserStmt->execute([$payout, $userId]);

            $result['updated'][] = [
                'bet_id'  => $betId,
                'user_id' => $userId,
                'payout'  => $payout,
            ];
        }

        // Alle anderen verlieren
        $pdo->prepare("
            UPDATE bets
            SET status='lost', payout=0, settled_at=NOW()
            WHERE race_id=? AND horse_id<>? AND status='placed'
        ")->execute([$raceId, $winnerHorseId]);
    }

    // Rennen als settled markieren
    $pdo->prepare("
        UPDATE races
        SET settled_at=NOW(), winner_horse_id=?
        WHERE id=?
    ")->execute([$winnerHorseId, $raceId]);

    $pdo->commit();

    return $result;
}

// -----------------------------------------------------------------------------
// "Controller"-Bereich (rudimentär)
// -----------------------------------------------------------------------------

// CLI-Modus: php race.php settle <raceId> <winnerHorseId>
// CLI-Modus: php race.php show   <raceId>
if (php_sapi_name() === 'cli') {

    $action = $argv[1] ?? null;

    if ($action === 'settle') {
        $raceId        = isset($argv[2]) ? (int)$argv[2] : 0;
        $winnerHorseId = isset($argv[3]) ? (int)$argv[3] : 0;

        if ($raceId > 0 && $winnerHorseId > 0) {
            $info = settleRaceBets($pdo, $raceId, $winnerHorseId);
            echo "Race $raceId settled. Winner horse $winnerHorseId.\n";
            echo "Pool: {$info['pool']}\n";
            echo "On winner: {$info['on_winner']}\n";
            echo "PayoutFactor: {$info['payoutFactor']}\n";
            echo "Updated bets:\n";
            foreach ($info['updated'] as $u) {
                echo "- Bet {$u['bet_id']} -> User {$u['user_id']} got {$u['payout']}\n";
            }
        } else {
            echo "Usage: php race.php settle <raceId> <winnerHorseId>\n";
        }

        exit;
    }

    if ($action === 'show') {
        $raceId = isset($argv[2]) ? (int)$argv[2] : 0;
        if ($raceId > 0) {
            $race = getRace($pdo, $raceId);
            $bets = getBetsForRace($pdo, $raceId);

            if (!$race) {
                echo "Race not found.\n";
                exit;
            }

            echo "Race #{$race['id']} {$race['name']}\n";
            echo "Starts: {$race['starts_at']}\n";
            echo "Settled: " . ($race['settled_at'] ?? 'not yet') . "\n";
            echo "Horses:\n";
            foreach ($race['horses'] as $h) {
                echo " - {$h['horse_id']}: {$h['horse_name']}\n";
            }
            echo "\nBets:\n";
            foreach ($bets as $b) {
                echo "Bet {$b['id']} User {$b['username']} ({$b['user_id']}) "
                   . "Horse {$b['horse_name']}({$b['horse_id']}) "
                   . "Amount {$b['amount']} Status {$b['status']} "
                   . "Payout {$b['payout']}\n";
            }
        } else {
            echo "Usage: php race.php show <raceId>\n";
        }

        exit;
    }

    echo "Usage:\n";
    echo "  php race.php settle <raceId> <winnerHorseId>\n";
    echo "  php race.php show   <raceId>\n";
    exit;
}


// -----------------------------------------------------------------------------
// HTTP-Modus (optional quick&dirty Admin-UI)
// -----------------------------------------------------------------------------

$action = $_GET['action'] ?? null;

if ($action === 'settle') {
    $raceId        = isset($_GET['race_id']) ? (int)$_GET['race_id'] : 0;
    $winnerHorseId = isset($_GET['winner_horse_id']) ? (int)$_GET['winner_horse_id'] : 0;

    if ($raceId > 0 && $winnerHorseId > 0) {
        $info = settleRaceBets($pdo, $raceId, $winnerHorseId);

        header('Content-Type: application/json');
        echo json_encode([
            'ok'            => true,
            'race_id'       => $raceId,
            'winner_horse'  => $winnerHorseId,
            'pool'          => $info['pool'],
            'on_winner'     => $info['on_winner'],
            'payout_factor' => $info['payoutFactor'],
            'updated'       => $info['updated'],
        ], JSON_PRETTY_PRINT);
    } else {
        http_response_code(400);
        echo json_encode(['ok'=>false,'error'=>'missing race_id or winner_horse_id']);
    }
    exit;
}

if ($action === 'show') {
    $raceId = isset($_GET['race_id']) ? (int)$_GET['race_id'] : 0;
    if ($raceId > 0) {
        $race = getRace($pdo, $raceId);
        $bets = getBetsForRace($pdo, $raceId);

        header('Content-Type: application/json');
        echo json_encode([
            'ok'   => (bool)$race,
            'race' => $race,
            'bets' => $bets,
        ], JSON_PRETTY_PRINT);
    } else {
        http_response_code(400);
        echo json_encode(['ok'=>false,'error'=>'missing race_id']);
    }
    exit;
}

// Fallback HTTP output
header('Content-Type: text/plain');
echo "race.php online.\n";
echo "Use:\n";
echo "  ?action=show&race_id=12\n";
echo "  ?action=settle&race_id=12&winner_horse_id=3\n";
?>
