<?php
declare(strict_types=1);
ini_set('display_errors', '1');
error_reporting(E_ALL);
session_start();

/**
 * Pferderennen v2 (PHP + MySQL, 1 Datei)
 * Features:
 * - Nutzer + virtuelles Geld (Login per Name, kein Passwort, Demo)
 * - Wetten (pari-mutuel, Einsatz wird sofort abgebucht, Gewinne nach Rennen gutgeschrieben)
 * - Rennen pending → Wetten per Formular → Start → Ergebnis → Animation (CSS/JS)
 * - Historie pro Pferd (Wins/Losses + letzte Ergebnisse)
 * - Admin-Bereich (Rennen/Pferde verwalten)
 *
 * ACHTUNG: Demo-Code (ohne CSRF, ohne Passwort), nur zu Testzwecken.
 */

function db(): PDO {
    static $pdo = null;
    if ($pdo) return $pdo;

    $dsn  = 'mysql:host=localhost;dbname=horses;charset=utf8';
    $user = 'root';        // <- anpassen
    $pass = '000000';    // <- anpassen

    $pdo = new PDO($dsn, $user, $pass, [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES   => false,
    ]);
    return $pdo;
}

function colExists(string $table, string $col): bool {
    $pdo = db();
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
    $stmt->execute([$table, $col]);
    return (bool)$stmt->fetchColumn();
}

function tableExists(string $table): bool {
    $pdo = db();
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
    $stmt->execute([$table]);
    return (bool)$stmt->fetchColumn();
}

function migrate(): void {
    $pdo = db();

    // horses
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS horses (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(100) NOT NULL,
            base_speed FLOAT NOT NULL,      -- m/s
            acceleration FLOAT NOT NULL,    -- m/s^2
            stamina TINYINT UNSIGNED NOT NULL,  -- 0..100
            temperament FLOAT NOT NULL,     -- 0..1
            active TINYINT(1) NOT NULL DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");

    // races
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS races (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            title VARCHAR(150) NOT NULL,
            track_length INT UNSIGNED NOT NULL,
            weather ENUM('clear','rain','wind','mud') NOT NULL DEFAULT 'clear',
            tick_ms INT UNSIGNED NOT NULL DEFAULT 200,
            status ENUM('pending','running','finished') NOT NULL DEFAULT 'finished',
            started_at DATETIME NULL,
            finished_at DATETIME NULL,
            settled_at DATETIME NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");
    // Upgrade alter columns if missing
    if (!colExists('races', 'status')) {
        $pdo->exec("ALTER TABLE races ADD COLUMN status ENUM('pending','running','finished') NOT NULL DEFAULT 'finished' AFTER tick_ms");
    }
    if (!colExists('races', 'settled_at')) {
        $pdo->exec("ALTER TABLE races ADD COLUMN settled_at DATETIME NULL AFTER finished_at");
    }

    // race_entries
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS race_entries (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            race_id INT UNSIGNED NOT NULL,
            horse_id INT UNSIGNED NOT NULL,
            lane TINYINT UNSIGNED DEFAULT NULL,
            distance FLOAT NOT NULL DEFAULT 0,
            velocity FLOAT NOT NULL DEFAULT 0,
            finished TINYINT(1) NOT NULL DEFAULT 0,
            finish_time_ms INT UNSIGNED DEFAULT NULL,
            place TINYINT UNSIGNED DEFAULT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX (race_id),
            INDEX (horse_id),
            CONSTRAINT fk_re_race FOREIGN KEY (race_id) REFERENCES races(id) ON DELETE CASCADE,
            CONSTRAINT fk_re_horse FOREIGN KEY (horse_id) REFERENCES horses(id) ON DELETE RESTRICT
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");

    // users
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS users (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            name VARCHAR(80) NOT NULL UNIQUE,
            balance DECIMAL(12,2) NOT NULL DEFAULT 1000.00,
            is_admin TINYINT(1) NOT NULL DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");

    // bets
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS bets (
            id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            race_id INT UNSIGNED NOT NULL,
            user_id INT UNSIGNED NOT NULL,
            horse_id INT UNSIGNED NOT NULL,
            amount DECIMAL(12,2) NOT NULL,
            status ENUM('placed','won','lost','void','settled') NOT NULL DEFAULT 'placed',
            payout DECIMAL(12,2) NOT NULL DEFAULT 0.00,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            settled_at DATETIME NULL,
            INDEX (race_id),
            INDEX (user_id),
            INDEX (horse_id),
            CONSTRAINT fk_b_race FOREIGN KEY (race_id) REFERENCES races(id) ON DELETE CASCADE,
            CONSTRAINT fk_b_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            CONSTRAINT fk_b_horse FOREIGN KEY (horse_id) REFERENCES horses(id) ON DELETE RESTRICT
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");

    // race_logs (ein JSON per Rennen für Animation)
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS race_logs (
            race_id INT UNSIGNED PRIMARY KEY,
            timeline MEDIUMTEXT NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            CONSTRAINT fk_rl_race FOREIGN KEY (race_id) REFERENCES races(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");
}

function seedHorsesIfEmpty(): void {
    $pdo = db();
    $count = (int)$pdo->query("SELECT COUNT(*) FROM horses")->fetchColumn();
    if ($count >= 12) return;

    $names = [
        'Blitz', 'Donner', 'Sturmwind', 'Nachtschatten', 'Himmelsläufer',
        'Feuerherz', 'Nordstern', 'Kupferkralle', 'Silbermond', 'Wolkenflitzer',
        'Schattenfell', 'Flammenläufer', 'Wellenreiter', 'Granithuf', 'Sonnenstrahl'
    ];
    shuffle($names);

    $stmt = $pdo->prepare("
        INSERT INTO horses (name, base_speed, acceleration, stamina, temperament, active)
        VALUES (?, ?, ?, ?, ?, 1)
    ");

    for ($i = 0; $i < 15; $i++) {
        $name = $names[$i % count($names)] . ' #' . ($i+1);
        $base = mt_rand(1400, 1800) / 100.0; // 14..18 m/s
        $acc  = mt_rand(35, 70) / 100.0;     // 0.35..0.70 m/s²
        $sta  = mt_rand(55, 100);
        $temp = mt_rand(10, 50) / 100.0;     // 0.10..0.50
        $stmt->execute([$name, $base, $acc, $sta, $temp]);
    }
}

function currentUser(): ?array {
    if (!isset($_SESSION['uid'])) return null;
    $pdo = db();
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id=?");
    $stmt->execute([$_SESSION['uid']]);
    return $stmt->fetch() ?: null;
}

function requireLogin(): array {
    $u = currentUser();
    if (!$u) {
        header("Location: ?action=login");
        exit;
    }
    return $u;
}

function requireAdmin(): array {
    $u = requireLogin();
    if (!$u['is_admin']) {
        http_response_code(403);
        echo "<p>Nur für Admins. <a href='?'>Zurück</a></p>";
        exit;
    }
    return $u;
}

function weatherFactors(string $weather): array {
    switch ($weather) {
        case 'rain': return [0.98, 0.90];
        case 'mud':  return [0.90, 0.85];
        case 'wind': return [0.95, 0.95];
        default:     return [1.00, 1.00]; // clear
    }
}

function clampInt(int $v, int $min, int $max): int {
    return max($min, min($max, $v));
}

function createPendingRace(int $n = 6, int $trackLen = 1200, string $weather = 'clear'): int {
    $pdo = db();
    seedHorsesIfEmpty();

    $n = clampInt($n, 2, 12);
    $trackLen = clampInt($trackLen, 600, 3200);
    $allowedWeather = ['clear','rain','wind','mud'];
    if (!in_array($weather, $allowedWeather, true)) $weather = 'clear';

    $horses = $pdo->query("SELECT * FROM horses WHERE active=1 ORDER BY RAND() LIMIT $n")->fetchAll();
    if (count($horses) < $n) {
        throw new RuntimeException("Nicht genügend aktive Pferde.");
    }

    $pdo->beginTransaction();
    $title = 'Rennen (pending) ' . date('Y-m-d H:i:s');
    $tickMs = 200;
    $stmtRace = $pdo->prepare("INSERT INTO races (title, track_length, weather, tick_ms, status) VALUES (?, ?, ?, ?, 'pending')");
    $stmtRace->execute([$title, $trackLen, $weather, $tickMs]);
    $raceId = (int)$pdo->lastInsertId();

    $stmtEntry = $pdo->prepare("INSERT INTO race_entries (race_id, horse_id, lane) VALUES (?, ?, ?)");
    $lane = 1;
    foreach ($horses as $h) {
        $stmtEntry->execute([$raceId, (int)$h['id'], $lane++]);
    }
    $pdo->commit();
    return $raceId;
}

// Simulation für ein EXISTIERENDES Rennen (mit vorhandenen Entries)
function simulateRace(int $raceId, bool $captureTimeline = true): void {
    $pdo = db();

    // Lade Rennen + Entries + Pferde
    $rStmt = $pdo->prepare("SELECT * FROM races WHERE id=?");
    $rStmt->execute([$raceId]);
    $race = $rStmt->fetch();
    if (!$race) throw new RuntimeException("Rennen nicht gefunden.");
    if ($race['status'] !== 'pending') throw new RuntimeException("Rennen hat Status {$race['status']} und kann nicht gestartet werden.");

    $eStmt = $pdo->prepare("
        SELECT re.*, h.name, h.base_speed, h.acceleration, h.stamina, h.temperament
        FROM race_entries re
        JOIN horses h ON h.id = re.horse_id
        WHERE re.race_id = ?
        ORDER BY re.lane ASC
    ");
    $eStmt->execute([$raceId]);
    $entries = $eStmt->fetchAll();
    if (count($entries) < 2) throw new RuntimeException("Zu wenige Teilnehmer.");

    $pdo->prepare("UPDATE races SET status='running', started_at=NOW() WHERE id=?")->execute([$raceId]);

    $dt = ((int)$race['tick_ms']) / 1000.0;
    [$speedF, $accF] = weatherFactors($race['weather']);
    $trackLen = (int)$race['track_length'];

    // States + Timeline (für Animation)
    $states = [];
    $remaining = count($entries);
    $timeline = []; // pro Entry: ds (array von Distanzen)
    foreach ($entries as $e) {
        $id = (int)$e['id'];
        $states[$id] = ['d'=>0.0, 'v'=>0.0, 'f'=>false, 't'=>null];
        $timeline[$id] = [];
    }

    $tick = 0;
    $maxTicks = 2000;
    $order = [];

    while ($remaining > 0 && $tick < $maxTicks) {
        foreach ($entries as $e) {
            $id = (int)$e['id'];
            if ($states[$id]['f']) {
                if ($captureTimeline) $timeline[$id][] = $states[$id]['d'];
                continue;
            }
            $h = $e;
            $dist = $states[$id]['d'];
            $vel  = $states[$id]['v'];
            $norm = $dist / $trackLen; // 0..1
            $fatigue = max(0.0, $norm - (float)$h['stamina'] / 120.0);
            $cap = (float)$h['base_speed'] * $speedF * (1.0 - min(0.7, $fatigue * 0.9));
            $target = min($cap, $vel + (float)$h['acceleration'] * $accF * $dt);

            $randUnit = (mt_rand() / mt_getrandmax()) * 2 - 1; // -1..1
            $noise = $randUnit * (0.35 * (float)$h['temperament']);

            $newV = max(0.0, min($cap, $target + $noise));
            $newD = $dist + $newV * $dt;

            if ($newD >= $trackLen) {
                $newD = (float)$trackLen;
                $states[$id]['d'] = $newD;
                $states[$id]['v'] = $newV;
                $states[$id]['f'] = true;
                $states[$id]['t'] = (int)round(($tick + 1) * $dt * 1000);
                $order[] = $id;
                $remaining--;
            } else {
                $states[$id]['d'] = $newD;
                $states[$id]['v'] = $newV;
            }
            if ($captureTimeline) $timeline[$id][] = $states[$id]['d'];
        }
        $tick++;
    }

    if ($remaining > 0) {
        // Nicht fertig: nach Distanz ordnen
        $still = [];
        foreach ($entries as $e) {
            $id = (int)$e['id'];
            if (!$states[$id]['f']) {
                $still[] = [$id, $states[$id]['d']];
            }
        }
        usort($still, fn($a,$b) => $b[1] <=> $a[1]);
        foreach ($still as [$id, $_]) {
            $states[$id]['f'] = true;
            $states[$id]['t'] = (int)round($tick * $dt * 1000) + 1;
            $order[] = $id;
        }
    }

    // Plätze vergeben
    $placeOf = [];
    $p = 1;
    foreach ($order as $id) $placeOf[$id] = $p++;

    // DB-Update
    $pdo->beginTransaction();
    $stmtUpd = $pdo->prepare("
        UPDATE race_entries
        SET distance=?, velocity=?, finished=?, finish_time_ms=?, place=?
        WHERE id=?
    ");
    foreach ($entries as $e) {
        $id = (int)$e['id'];
        $st = $states[$id];
        $stmtUpd->execute([
            $st['d'], $st['v'], $st['f'] ? 1 : 0, $st['t'], $placeOf[$id] ?? null, $id
        ]);
    }
    $pdo->prepare("UPDATE races SET status='finished', finished_at=NOW() WHERE id=?")->execute([$raceId]);
    $pdo->commit();

    // Timeline speichern
    if ($captureTimeline) {
        // Struktur: { dt, len, entries: [ {entry_id, name, lane, color, ds:[]}, ... ] }
        $colors = ['#d7263d','#1b998b','#2e294e','#f46036','#e2c044','#0e7c7b','#8d5b4c','#6a4c93','#3a86ff','#ff006e','#8338ec','#fb5607'];
        $payload = [
            'dt' => $dt,
            'len' => $trackLen,
            'entries' => []
        ];
        $i=0;
        foreach ($entries as $e) {
            $eid = (int)$e['id'];
            $payload['entries'][] = [
                'entry_id' => $eid,
                'name'     => $e['name'],
                'lane'     => (int)$e['lane'],
                'color'    => $colors[$i % count($colors)],
                'ds'       => array_map(fn($v)=>round($v,3), $timeline[$eid] ?? [])
            ];
            $i++;
        }
        $json = json_encode($payload, JSON_UNESCAPED_UNICODE);
        $stmt = $pdo->prepare("INSERT INTO race_logs (race_id, timeline) VALUES (?, ?) ON DUPLICATE KEY UPDATE timeline=VALUES(timeline)");
        $stmt->execute([$raceId, $json]);
    }

    // Nach Zieleinlauf: Wetten auszahlen (falls noch nicht gesettled)
    settleRaceBets($raceId);
}

function settleRaceBets(int $raceId): void {
    $pdo = db();
    // Bereits gesettled?
    $r = $pdo->prepare("SELECT id, settled_at FROM races WHERE id=?");
    $r->execute([$raceId]);
    $race = $r->fetch();
    if (!$race) return;
    if ($race['settled_at']) return;

    // Gewinnerpferd (Platz 1)
    $win = $pdo->prepare("SELECT horse_id FROM race_entries WHERE race_id=? AND place=1 LIMIT 1");
    $win->execute([$raceId]);
    $winnerHorseId = $win->fetchColumn();
    if (!$winnerHorseId) {
        $pdo->prepare("UPDATE races SET settled_at=NOW() WHERE id=?")->execute([$raceId]);
        return;
    }

    // Pool und Einsätze
    $pool = (float)$pdo->prepare("SELECT IFNULL(SUM(amount),0) FROM bets WHERE race_id=? AND status='placed'")->execute([$raceId]) ?: 0.0;
    $stmtPool = $pdo->prepare("SELECT IFNULL(SUM(amount),0) FROM bets WHERE race_id=? AND horse_id=? AND status='placed'");
    $stmtPool->execute([$raceId, $winnerHorseId]);
    $onWinner = (float)$stmtPool->fetchColumn();

    $pdo->beginTransaction();

    if ($pool <= 0.0) {
        // Nichts zu tun
    } elseif ($onWinner <= 0.0) {
        // Keiner hat auf den Gewinner gewettet -> alle verlieren, Haus behält Pool
        $pdo->prepare("UPDATE bets SET status='lost', payout=0, settled_at=NOW() WHERE race_id=? AND status='placed'")
            ->execute([$raceId]);
    } else {
        $payoutFactor = $pool / $onWinner; // pari-mutuel, ohne Hausanteil
        // Gewinner
        $winners = $pdo->prepare("SELECT id, user_id, amount FROM bets WHERE race_id=? AND horse_id=? AND status='placed'");
        $winners->execute([$raceId, $winnerHorseId]);
        $updBet = $pdo->prepare("UPDATE bets SET status='won', payout=?, settled_at=NOW() WHERE id=?");
        $updUser= $pdo->prepare("UPDATE users SET balance=balance+? WHERE id=?");
        foreach ($winners as $b) {
            $payout = round((float)$b['amount'] * $payoutFactor, 2);
            $updBet->execute([$payout, (int)$b['id']]);
            $updUser->execute([$payout, (int)$b['user_id']]);
        }
        // Verlierer
        $pdo->prepare("UPDATE bets SET status='lost', payout=0, settled_at=NOW() WHERE race_id=? AND horse_id<>? AND status='placed'")
            ->execute([$raceId, $winnerHorseId]);
    }

    $pdo->prepare("UPDATE races SET settled_at=NOW() WHERE id=?")->execute([$raceId]);
    $pdo->commit();
}

function formatMoney($v): string {
    return number_format((float)$v, 2, ',', '.') . ' €';
}

/* ---------- Rendering ---------- */

function baseHeader(string $title = 'Pferderennen'): string {
    $u = currentUser();
    $userHtml = $u ? ("Angemeldet: <strong>".htmlspecialchars($u['name'])."</strong> · Guthaben: <strong>".formatMoney($u['balance'])."</strong> ".
        ($u['is_admin'] ? "· <span style='color:#0a0'>Admin</span>" : "").
        " · <a href='?action=logout'>Logout</a>") : "<a href='?action=login'>Login</a>";
    return '<!doctype html><html lang="de"><head><meta charset="utf-8"><meta name="viewport" content="width=device-width, initial-scale=1">
    <title>'.htmlspecialchars($title).'</title>
    <style>
    body{font-family:system-ui,Segoe UI,Roboto,Arial,sans-serif;margin:24px;color:#222}
    a.btn{display:inline-block;margin:6px 8px;padding:8px 12px;border:1px solid #ddd;border-radius:6px;text-decoration:none;color:#222;background:#f9f9f9}
    a.btn:hover{background:#fff}
    table{border-collapse:collapse;width:100%;max-width:980px}
    th,td{padding:8px 10px;border-bottom:1px solid #eee;text-align:left}
    th{background:#fafafa}
    .meta{color:#555;margin-bottom:10px}
    .mono{font-family:ui-monospace,Menlo,Consolas,monospace}
    .track{position:relative;border:1px solid #ddd;border-radius:8px;background:#f7f7f7;height:320px;max-width:980px;overflow:hidden}
    .lane{position:absolute;left:0;right:0;height:24px;border-bottom:1px dashed #e0e0e0}
    .horse{position:absolute;left:0;transform:translateX(0);top:0;height:18px;width:18px;border-radius:50%}
    .controls button{margin-right:6px}
    .pill{display:inline-block;padding:2px 8px;border-radius:999px;background:#eee}
    .warn{color:#b00}
    </style></head><body>
    <div class="meta">'.$userHtml.' · <a class="btn" href="?">Home</a> '.($u && $u['is_admin'] ? '· <a class="btn" href="?action=admin">Admin</a>' : '').'</div>';
}

function baseFooter(): string {
    return '</body></html>';
}

function renderHome(): string {
    seedHorsesIfEmpty();
    $pdo = db();
    $lastId = $pdo->query("SELECT id FROM races ORDER BY id DESC LIMIT 1")->fetchColumn();
    $html = baseHeader('Pferderennen');
    $html .= '<h1>🏇 Pferderennen</h1>';
    $html .= '<p>Lege ein Rennen an (pending), platziere Wetten und starte das Rennen.</p>';
    $html .= '<p>
        <a class="btn" href="?action=migrate">DB-Migration</a>
        <a class="btn" href="?action=login">Login</a>
        <a class="btn" href="?action=create&n=8&len=1200&weather=clear">Rennen anlegen (Admin)</a>
        '.($lastId ? '<a class="btn" href="?action=view&id='.$lastId.'">Letztes Rennen ansehen</a>' : '').'
    </p>';

    // Letzte Rennen
    $rows = $pdo->query("SELECT id, title, status, weather, track_length, started_at, finished_at FROM races ORDER BY id DESC LIMIT 10")->fetchAll();
    $html .= '<h2>Letzte Rennen</h2><table><thead><tr><th>ID</th><th>Titel</th><th>Status</th><th>Strecke</th><th>Wetter</th><th>Start</th><th>Ende</th><th>Aktion</th></tr></thead><tbody>';
    foreach ($rows as $r) {
        $html .= '<tr>'.
            '<td>'.$r['id'].'</td>'.
            '<td>'.htmlspecialchars($r['title']).'</td>'.
            '<td><span class="pill">'.$r['status'].'</span></td>'.
            '<td>'.$r['track_length'].' m</td>'.
            '<td>'.$r['weather'].'</td>'.
            '<td>'.($r['started_at'] ?: '—').'</td>'.
            '<td>'.($r['finished_at'] ?: '—').'</td>'.
            '<td>'.
                '<a class="btn" href="?action=view&id='.$r['id'].'">View</a> '.
                ($r['status']==='pending' ? '<a class="btn" href="?action=bet&id='.$r['id'].'">Wetten</a>' : '').
            '</td>'.
        '</tr>';
    }
    $html .= '</tbody></table>';
    $html .= baseFooter();
    return $html;
}

function renderLogin(): string {
    $u = currentUser();
    $html = baseHeader('Login');
    $html .= '<h2>Login</h2>';
    if ($u) {
        $html .= '<p>Bereits angemeldet als <strong>'.htmlspecialchars($u['name']).'</strong>.</p>';
    }
    $html .= '<form method="post" action="?action=do_login">
        <label>Name: <input required name="name" maxlength="80"></label>
        <button type="submit">Anmelden</button>
    </form>
    <p style="color:#666">Hinweis: Demo-Login ohne Passwort. Der erste User wird automatisch Admin.</p>';
    $html .= baseFooter();
    return $html;
}

function handleDoLogin(): void {
    $name = trim($_POST['name'] ?? '');
    if ($name === '') {
        header("Location: ?action=login");
        exit;
    }
    $pdo = db();
    $pdo->beginTransaction();
    $stmt = $pdo->prepare("SELECT * FROM users WHERE name=?");
    $stmt->execute([$name]);
    $user = $stmt->fetch();
    if (!$user) {
        // erster User wird Admin
        $isFirst = !(bool)$pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
        $ins = $pdo->prepare("INSERT INTO users (name, balance, is_admin) VALUES (?, 1000.00, ?)");
        $ins->execute([$name, $isFirst ? 1 : 0]);
        $userId = (int)$pdo->lastInsertId();
        $user = $pdo->query("SELECT * FROM users WHERE id=".$userId)->fetch();
    }
    $pdo->commit();
    $_SESSION['uid'] = (int)$user['id'];
    header("Location: ?");
    exit;
}

function handleLogout(): void {
    session_destroy();
    header("Location: ?");
    exit;
}

function renderBetForm(int $raceId): string {
    $pdo = db();
    $r = $pdo->prepare("SELECT * FROM races WHERE id=?");
    $r->execute([$raceId]);
    $race = $r->fetch();
    if (!$race) return "<p>Rennen nicht gefunden. <a href='?'>Zurück</a></p>";
    if ($race['status'] !== 'pending') return "<p>Wetten nicht möglich. Rennen ist {$race['status']}.</p>";

    $u = requireLogin();

    $e = $pdo->prepare("
        SELECT re.id as entry_id, re.lane, h.id as horse_id, h.name
        FROM race_entries re JOIN horses h ON h.id = re.horse_id
        WHERE re.race_id=? ORDER BY re.lane
    ");
    $e->execute([$raceId]);
    $rows = $e->fetchAll();

    // Pools
    $poolTotal = (float)$pdo->prepare("SELECT IFNULL(SUM(amount),0) FROM bets WHERE race_id=?")
        ->execute([$raceId]) ?: 0.0;
    $stmtHorsePool = $pdo->prepare("SELECT IFNULL(SUM(amount),0) FROM bets WHERE race_id=? AND horse_id=?");

    $html = baseHeader('Wetten');
    $html .= '<h2>Wetten platzieren</h2>';
    $html .= '<div class="meta">Rennen #'.$race['id'].' · Strecke: '.$race['track_length'].' m · Wetter: '.$race['weather'].' · Status: '.$race['status'].'</div>';

    $html .= '<form method="post" action="?action=place_bet&id='.$raceId.'"><table><thead><tr><th>Lane</th><th>Pferd</th><th>Dein Einsatz (€)</th><th>Pool</th></tr></thead><tbody>';
    foreach ($rows as $row) {
        $stmtHorsePool->execute([$raceId, $row['horse_id']]);
        $hp = (float)$stmtHorsePool->fetchColumn();
        $html .= '<tr>'.
            '<td>'.(int)$row['lane'].'</td>'.
            '<td>'.htmlspecialchars($row['name']).'</td>'.
            '<td><input type="number" name="bet['.$row['horse_id'].']" step="0.01" min="0" placeholder="0.00" style="width:120px"></td>'.
            '<td>'.formatMoney($hp).'</td>'.
        '</tr>';
    }
    $html .= '</tbody></table><p>Guthaben: <strong>'.formatMoney($u['balance']).'</strong></p>';
    $html .= '<button type="submit">Wette platzieren</button> ';
    $html .= '<a class="btn" href="?action=view&id='.$raceId.'">Zum Rennen</a>';
    $html .= '</form>';

    $html .= baseFooter();
    return $html;
}

function handlePlaceBet(int $raceId): string {
    $pdo = db();
    $u = requireLogin();
    // Race prüfen
    $r = $pdo->prepare("SELECT * FROM races WHERE id=?");
    $r->execute([$raceId]);
    $race = $r->fetch();
    if (!$race) return "<p>Rennen nicht gefunden.</p>";
    if ($race['status'] !== 'pending') return "<p>Wetten geschlossen.</p>";

    $bets = $_POST['bet'] ?? [];
    // Filter > 0
    $toPlace = [];
    $total = 0.0;
    foreach ($bets as $hid => $amt) {
        $amt = (float)$amt;
        if ($amt > 0) {
            $toPlace[(int)$hid] = $amt;
            $total += $amt;
        }
    }
    if (!$toPlace) {
        header("Location: ?action=bet&id=".$raceId);
        exit;
    }
    // Guthaben prüfen
    $stmtBal = $pdo->prepare("SELECT balance FROM users WHERE id=?");
    $stmtBal->execute([$u['id']]);
    $balance = (float)$stmtBal->fetchColumn();
    if ($total > $balance + 1e-9) {
        return "<p class='warn'>Nicht genügend Guthaben. Benötigt ".formatMoney($total).", vorhanden ".formatMoney($balance).". <a href='?action=bet&id=$raceId'>Zurück</a></p>";
    }

    // Horse IDs valid?
    $validHids = $pdo->prepare("SELECT horse_id FROM race_entries WHERE race_id=?");
    $validHids->execute([$raceId]);
    $validSet = array_flip(array_map('intval', array_column($validHids->fetchAll(), 'horse_id')));

    $pdo->beginTransaction();
    $ins = $pdo->prepare("INSERT INTO bets (race_id, user_id, horse_id, amount) VALUES (?, ?, ?, ?)");
    foreach ($toPlace as $hid => $amt) {
        if (!isset($validSet[$hid])) continue;
        $ins->execute([$raceId, (int)$u['id'], (int)$hid, round($amt, 2)]);
    }
    // Guthaben abziehen
    $pdo->prepare("UPDATE users SET balance = balance - ? WHERE id=?")->execute([round($total,2), (int)$u['id']]);
    $pdo->commit();

    header("Location: ?action=bet&id=".$raceId);
    exit;
}

function renderViewRace(int $raceId): string {
    $pdo = db();
    $r = $pdo->prepare("SELECT * FROM races WHERE id=?");
    $r->execute([$raceId]);
    $race = $r->fetch();
    if (!$race) return "<p>Rennen nicht gefunden. <a href='?'>Zurück</a></p>";

    $entries = $pdo->prepare("
        SELECT re.*, h.name
        FROM race_entries re JOIN horses h ON h.id = re.horse_id
        WHERE re.race_id=? ORDER BY IFNULL(re.place, 999), re.finish_time_ms, re.lane
    ");
    $entries->execute([$raceId]);
    $rows = $entries->fetchAll();

    // Pool + eigene Wetten
    $pool = (float)$pdo->query("SELECT IFNULL(SUM(amount),0) FROM bets WHERE race_id=".$raceId."")->fetchColumn();
    $me = currentUser();
    $myBets = [];
    if ($me) {
        $b = $pdo->prepare("
            SELECT b.*, h.name AS hname FROM bets b
            JOIN horses h ON h.id=b.horse_id WHERE b.race_id=? AND b.user_id=? ORDER BY b.created_at
        ");
        $b->execute([$raceId, (int)$me['id']]);
        $myBets = $b->fetchAll();
    }

    $html = baseHeader('Rennen ansehen');
    $html .= '<h2>Rennen #'.$race['id'].'</h2>';
    $html .= '<div class="meta">Status: '.$race['status'].' · Strecke: '.$race['track_length'].' m · Wetter: '.$race['weather'].' · Pool: '.formatMoney($pool).'</div>';

    // Buttons
    $html .= '<p>';
    if ($race['status'] === 'pending') {
        $html .= '<a class="btn" href="?action=bet&id='.$raceId.'">Wetten</a> ';
    }
    if ($me && $me['is_admin'] && $race['status'] === 'pending') {
        $html .= '<a class="btn" href="?action=start&id='.$raceId.'">Rennen starten (Admin)</a> ';
    }
    if ($race['status'] === 'finished') {
        $html .= '<a class="btn" href="?action=watch&id='.$raceId.'">Animation abspielen</a> ';
    }
    $html .= '</p>';

    // Tabelle
    $html .= '<table><thead><tr><th>Platz</th><th>Pferd</th><th>Zeit (s)</th><th>Lane</th><th>Distanz</th></tr></thead><tbody>';
    foreach ($rows as $r2) {
        $timeS = $r2['finish_time_ms'] ? number_format(((int)$r2['finish_time_ms'])/1000.0, 2) : '—';
        $html .= '<tr>'.
            '<td>'.($r2['place'] ? (int)$r2['place'] : '—').'</td>'.
            '<td><a href="?action=horse&id='.$r2['horse_id'].'">'.htmlspecialchars($r2['name']).'</a></td>'.
            '<td>'.$timeS.'</td>'.
            '<td>'.$r2['lane'].'</td>'.
            '<td>'.number_format((float)$r2['distance'], 1).' m</td>'.
        '</tr>';
    }
    $html .= '</tbody></table>';

    if ($myBets) {
        $html .= '<h3>Deine Wetten</h3><table><thead><tr><th>Pferd</th><th>Einsatz</th><th>Status</th><th>Payout</th><th>Zeit</th></tr></thead><tbody>';
        foreach ($myBets as $b) {
            $html .= '<tr>'.
            '<td>'.htmlspecialchars($b['hname']).'</td>'.
            '<td>'.formatMoney($b['amount']).'</td>'.
            '<td>'.$b['status'].'</td>'.
            '<td>'.formatMoney($b['payout']).'</td>'.
            '<td>'.$b['created_at'].'</td>'.
            '</tr>';
        }
        $html .= '</tbody></table>';
    }

    $html .= baseFooter();
    return $html;
}

function renderWatch(int $raceId): string {
    $pdo = db();
    $r = $pdo->prepare("SELECT * FROM races WHERE id=?");
    $r->execute([$raceId]);
    $race = $r->fetch();
    if (!$race) return "<p>Rennen nicht gefunden.</p>";
    if ($race['status'] !== 'finished') return "<p>Rennen ist noch nicht beendet.</p>";

    $log = $pdo->prepare("SELECT timeline FROM race_logs WHERE race_id=?");
    $log->execute([$raceId]);
    $timeline = $log->fetchColumn();
    if (!$timeline) {
        $note = "<p>Keine Timeline gespeichert. Animation nicht möglich.</p>";
    } else {
        $note = "";
    }

    $html = baseHeader('Animation');
    $html .= '<h2>Animation Rennen #'.$raceId.'</h2>';
    $html .= '<div class="meta">Strecke: '.$race['track_length'].' m · Wetter: '.$race['weather'].'</div>';
    $html .= $note;

    $html .= '<div id="track" class="track"></div>
    <div class="controls" style="margin-top:8px;">
        <button id="btnPlay">Play</button>
        <button id="btnPause">Pause</button>
        <button id="btnReset">Reset</button>
        <span id="tLabel" class="pill">t = 0.0 s</span>
    </div>
    <script>
    const TL = '.($timeline ? json_encode(json_decode($timeline)) : 'null').';
    const track = document.getElementById("track");
    const btnPlay = document.getElementById("btnPlay");
    const btnPause = document.getElementById("btnPause");
    const btnReset = document.getElementById("btnReset");
    const tLabel = document.getElementById("tLabel");

    if (TL) {
        // Layout: eine Lane pro Entry
        const lanes = TL.entries.length;
        const laneH = 26;
        track.style.height = (laneH * lanes + 12) + "px";
        const W = track.clientWidth - 40; // Padding rechts für Rand
        const horses = [];

        TL.entries.forEach((e, idx) => {
            const y = idx * laneH + 6;
            const l = document.createElement("div");
            l.className = "lane";
            l.style.top = y + "px";
            track.appendChild(l);

            const h = document.createElement("div");
            h.className = "horse";
            h.style.top = (y+3) + "px";
            h.style.background = e.color;
            h.title = e.name + " (Lane "+e.lane+")";
            track.appendChild(h);

            // Label
            const label = document.createElement("div");
            label.style.position = "absolute";
            label.style.left = "6px";
            label.style.top = (y-12) + "px";
            label.style.fontSize = "12px";
            label.textContent = e.name;
            track.appendChild(label);

            horses.push({el:h, ds:e.ds});
        });

        let t = 0; // frame index
        let raf = null;
        let playing = false;
        const dt = TL.dt;
        const len = TL.len;
        const maxFrames = Math.max(...horses.map(h => h.ds.length));

        function renderFrame(i) {
            horses.forEach(h => {
                const d = h.ds[Math.min(i, h.ds.length-1)] || 0;
                const x = Math.min(1, d / len) * W;
                h.el.style.transform = "translateX("+x+"px)";
            });
            tLabel.textContent = "t = " + (i*dt).toFixed(1) + " s";
        }
        function step() {
            if (!playing) return;
            renderFrame(t);
            t++;
            if (t >= maxFrames) { playing = false; cancelAnimationFrame(raf); return; }
            raf = requestAnimationFrame(step);
        }
        btnPlay.onclick = () => { if (!playing){ playing=true; step(); } };
        btnPause.onclick = () => { playing=false; cancelAnimationFrame(raf); };
        btnReset.onclick = () => { playing=false; cancelAnimationFrame(raf); t=0; renderFrame(0); };
        renderFrame(0);
    }
    </script>';

    $html .= baseFooter();
    return $html;
}

function renderHorse(int $horseId): string {
    $pdo = db();
    $h = $pdo->prepare("SELECT * FROM horses WHERE id=?");
    $h->execute([$horseId]);
    $horse = $h->fetch();
    if (!$horse) return "<p>Pferd nicht gefunden.</p>";

    // Historie aggregieren
    $win = $pdo->prepare("SELECT COUNT(*) FROM race_entries WHERE horse_id=? AND place=1");
    $win->execute([$horseId]);
    $wins = (int)$win->fetchColumn();
    $races = (int)$pdo->prepare("SELECT COUNT(*) FROM race_entries WHERE horse_id=?")->execute([$horseId]) ?: 0;
    $stmtR = $pdo->prepare("
        SELECT re.*, r.title, r.finished_at
        FROM race_entries re JOIN races r ON r.id=re.race_id
        WHERE re.horse_id=?
        ORDER BY r.id DESC LIMIT 10
    ");
    $stmtR->execute([$horseId]);
    $rows = $stmtR->fetchAll();

    $html = baseHeader('Pferd');
    $html .= '<h2>'.htmlspecialchars($horse['name']).'</h2>';
    $html .= '<div class="meta">Basis v='.number_format($horse['base_speed'],2).' m/s · a='.number_format($horse['acceleration'],2).' m/s² · Stamina='.$horse['stamina'].' · Temperament='.number_format($horse['temperament'],2).'</div>';
    $html .= '<p>Bilanz: <strong>'.$wins.' Siege</strong> aus <strong>'.$races.' Rennen</strong></p>';

    $html .= '<h3>Letzte Rennen</h3><table><thead><tr><th>Rennen</th><th>Datum</th><th>Platz</th><th>Zeit (s)</th><th>Distanz</th></tr></thead><tbody>';
    foreach ($rows as $r) {
        $t = $r['finish_time_ms'] ? number_format(((int)$r['finish_time_ms'])/1000.0, 2) : '—';
        $html .= '<tr>'.
        '<td><a href="?action=view&id='.$r['race_id'].'">'.htmlspecialchars($r['title']).'</a></td>'.
        '<td>'.($r['finished_at'] ?: '—').'</td>'.
        '<td>'.($r['place'] ?: '—').'</td>'.
        '<td>'.$t.'</td>'.
        '<td>'.number_format((float)$r['distance'], 1).' m</td>'.
        '</tr>';
    }
    $html .= '</tbody></table>';

    $html .= baseFooter();
    return $html;
}

/* ---------- Admin ---------- */

function renderAdmin(): string {
    $u = requireAdmin();
    $pdo = db();

    $horses = $pdo->query("SELECT * FROM horses ORDER BY id DESC LIMIT 50")->fetchAll();
    $races  = $pdo->query("SELECT * FROM races ORDER BY id DESC LIMIT 20")->fetchAll();
    $users  = $pdo->query("SELECT id, name, balance, is_admin, created_at FROM users ORDER BY id ASC")->fetchAll();

    $html = baseHeader('Admin');
    $html .= '<h2>Admin</h2>';

    // Aktionen
    $html .= '<p>
        <a class="btn" href="?action=create&n=8&len=1200&weather=clear">Rennen anlegen</a>
    </p>';

    // Horses
    $html .= '<h3>Pferde</h3><table><thead><tr><th>ID</th><th>Name</th><th>v</th><th>a</th><th>sta</th><th>temp</th><th>active</th><th>Aktion</th></tr></thead><tbody>';
    foreach ($horses as $h) {
        $html .= '<tr>'.
            '<td>'.$h['id'].'</td>'.
            '<td>'.$h['name'].'</td>'.
            '<td>'.number_format($h['base_speed'],2).'</td>'.
            '<td>'.number_format($h['acceleration'],2).'</td>'.
            '<td>'.$h['stamina'].'</td>'.
            '<td>'.number_format($h['temperament'],2).'</td>'.
            '<td>'.($h['active'] ? '1' : '0').'</td>'.
            '<td>'.
              '<a class="btn" href="?action=admin_edit_horse&id='.$h['id'].'">Bearbeiten</a> '.
              '<a class="btn" href="?action=admin_delete_horse&id='.$h['id'].'" onclick="return confirm(\'Wirklich löschen?\')">Löschen</a>'.
            '</td>'.
        '</tr>';
    }
    $html .= '</tbody></table>';

    // Races
    $html .= '<h3>Rennen</h3><table><thead><tr><th>ID</th><th>Titel</th><th>Status</th><th>Aktion</th></tr></thead><tbody>';
    foreach ($races as $r) {
        $html .= '<tr>'.
            '<td>'.$r['id'].'</td>'.
            '<td>'.htmlspecialchars($r['title']).'</td>'.
            '<td>'.$r['status'].'</td>'.
            '<td>'.
                ($r['status']==='pending' ? '<a class="btn" href="?action=start&id='.$r['id'].'">Start</a> ' : '').
                '<a class="btn" href="?action=view&id='.$r['id'].'">View</a> '.
                '<a class="btn" href="?action=admin_delete_race&id='.$r['id'].'" onclick="return confirm(\'Rennen löschen?\')">Löschen</a>'.
            '</td>'.
        '</tr>';
    }
    $html .= '</tbody></table>';

    // Users
    $html .= '<h3>Benutzer</h3><table><thead><tr><th>ID</th><th>Name</th><th>Guthaben</th><th>Admin</th><th>Aktion</th></tr></thead><tbody>';
    foreach ($users as $us) {
        $html .= '<tr>'.
            '<td>'.$us['id'].'</td>'.
            '<td>'.$us['name'].'</td>'.
            '<td>'.formatMoney($us['balance']).'</td>'.
            '<td>'.($us['is_admin'] ? 'ja' : 'nein').'</td>'.
            '<td>'.
                '<a class="btn" href="?action=admin_topup&id='.$us['id'].'&amt=100">+100€</a> '.
                '<a class="btn" href="?action=admin_toggle_admin&id='.$us['id'].'">'.($us['is_admin']?'Admin entziehen':'Admin geben').'</a>'.
            '</td>'.
        '</tr>';
    }
    $html .= '</tbody></table>';

    $html .= baseFooter();
    return $html;
}

function renderAdminEditHorse(int $horseId): string {
    $pdo = db();
    $h = $pdo->prepare("SELECT * FROM horses WHERE id=?");
    $h->execute([$horseId]);
    $horse = $h->fetch();
    if (!$horse) return "<p>Pferd nicht gefunden.</p>";

    requireAdmin();
    $html = baseHeader('Pferd bearbeiten');
    $html .= '<h2>Pferd bearbeiten</h2>';
    $html .= '<form method="post" action="?action=admin_save_horse&id='.$horseId.'">
        <label>Name: <input name="name" value="'.htmlspecialchars($horse['name']).'"></label><br>
        <label>Base v (m/s): <input step="0.01" type="number" name="base_speed" value="'.htmlspecialchars((string)$horse['base_speed']).'"></label><br>
        <label>a (m/s²): <input step="0.01" type="number" name="acceleration" value="'.htmlspecialchars((string)$horse['acceleration']).'"></label><br>
        <label>Stamina: <input type="number" name="stamina" min="0" max="100" value="'.(int)$horse['stamina'].'"></label><br>
        <label>Temperament: <input step="0.01" type="number" name="temperament" min="0" max="1" value="'.htmlspecialchars((string)$horse['temperament']).'"></label><br>
        <label>Active: <select name="active"><option value="1" '.($horse['active']?'selected':'').'>1</option><option value="0" '.(!$horse['active']?'selected':'').'>0</option></select></label><br>
        <button type="submit">Speichern</button>
    </form>';
    $html .= baseFooter();
    return $html;
}

function handleAdminSaveHorse(int $horseId): void {
    requireAdmin();
    $pdo = db();
    $stmt = $pdo->prepare("UPDATE horses SET name=?, base_speed=?, acceleration=?, stamina=?, temperament=?, active=? WHERE id=?");
    $stmt->execute([
        trim($_POST['name'] ?? ''),
        (float)($_POST['base_speed'] ?? 0),
        (float)($_POST['acceleration'] ?? 0),
        (int)($_POST['stamina'] ?? 0),
        (float)($_POST['temperament'] ?? 0),
        (int)($_POST['active'] ?? 1),
        $horseId
    ]);
    header("Location: ?action=admin");
    exit;
}

function handleAdminDeleteHorse(int $horseId): string {
    requireAdmin();
    $pdo = db();
    try {
        $stmt = $pdo->prepare("DELETE FROM horses WHERE id=?");
        $stmt->execute([$horseId]);
        header("Location: ?action=admin");
        exit;
    } catch (Throwable $e) {
        return "<p class='warn'>Löschen fehlgeschlagen (ggf. in Rennen verwendet). <a href='?action=admin'>Zurück</a></p>";
    }
}

function handleAdminDeleteRace(int $raceId): void {
    requireAdmin();
    $pdo = db();
    $pdo->prepare("DELETE FROM races WHERE id=?")->execute([$raceId]);
    header("Location: ?action=admin");
    exit;
}

function handleAdminTopup(int $userId, float $amt): void {
    requireAdmin();
    $pdo = db();
    $pdo->prepare("UPDATE users SET balance = balance + ? WHERE id=?")->execute([round($amt,2), $userId]);
    header("Location: ?action=admin");
    exit;
}

function handleAdminToggleAdmin(int $userId): void {
    requireAdmin();
    $pdo = db();
    $pdo->prepare("UPDATE users SET is_admin = 1 - is_admin WHERE id=?")->execute([$userId]);
    header("Location: ?action=admin");
    exit;
}

/* ---------- Router ---------- */

$action = $_GET['action'] ?? 'home';

try {
    switch ($action) {
        case 'migrate':
            migrate();
            echo "<p>Migration OK. <a href='?'>Home</a></p>";
            break;

        case 'login':
            echo renderLogin();
            break;
        case 'do_login':
            handleDoLogin();
            break;
        case 'logout':
            handleLogout();
            break;

        case 'home':
        default:
            echo renderHome();
            break;

        case 'create': { // Admin: pending race erstellen
            requireAdmin();
            $n = isset($_GET['n']) ? (int)$_GET['n'] : 8;
            $len = isset($_GET['len']) ? (int)$_GET['len'] : 1200;
            $weather = $_GET['weather'] ?? 'clear';
            $raceId = createPendingRace($n, $len, $weather);
            header("Location: ?action=bet&id=".$raceId);
            exit;
        }

        case 'bet': {
            $id = (int)($_GET['id'] ?? 0);
            echo renderBetForm($id);
            break;
        }
        case 'place_bet': {
            $id = (int)($_GET['id'] ?? 0);
            echo handlePlaceBet($id);
            break;
        }

        case 'start': {
            requireAdmin();
            $id = (int)($_GET['id'] ?? 0);
            simulateRace($id, true);
            header("Location: ?action=view&id=".$id);
            exit;
        }

        case 'view': {
            $id = (int)($_GET['id'] ?? 0);
            echo renderViewRace($id);
            break;
        }

        case 'watch': {
            $id = (int)($_GET['id'] ?? 0);
            echo renderWatch($id);
            break;
        }

        case 'horse': {
            $id = (int)($_GET['id'] ?? 0);
            echo renderHorse($id);
            break;
        }

        // Admin
        case 'admin':
            echo renderAdmin();
            break;
        case 'admin_edit_horse':
            echo renderAdminEditHorse((int)($_GET['id'] ?? 0));
            break;
        case 'admin_save_horse':
            handleAdminSaveHorse((int)($_GET['id'] ?? 0));
            break;
        case 'admin_delete_horse':
            echo handleAdminDeleteHorse((int)($_GET['id'] ?? 0));
            break;
        case 'admin_delete_race':
            handleAdminDeleteRace((int)($_GET['id'] ?? 0));
            break;
        case 'admin_topup':
            handleAdminTopup((int)($_GET['id'] ?? 0), (float)($_GET['amt'] ?? 0));
            break;
        case 'admin_toggle_admin':
            handleAdminToggleAdmin((int)($_GET['id'] ?? 0));
            break;
    }
} catch (Throwable $e) {
    http_response_code(500);
    echo "<pre style='color:#b00'>Fehler: " . htmlspecialchars($e->getMessage()) . "</pre>";
}
